<?php

namespace App\Http\Controllers\Admin;
use App\Models\User;
use App\Notifications\AspirasiMasukNotification;
use App\Http\Controllers\Controller;
use App\Models\Aspiration;
use Illuminate\Http\Request;
use App\Mail\AspirationSubmitted;
use Illuminate\Support\Facades\Mail;


class AspirationController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = Aspiration::orderBy('created_at', 'desc');

            if ($request->has('status') && $request->status != '') {
                $query->where('status', $request->status);
            }

            $aspirations = $query->paginate(20);

            return view('admin.aspirations.index', compact('aspirations'));
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    public function show(Aspiration $aspiration)
    {
        return view('admin.aspirations.show', compact('aspiration'));
    }

    public function update(Request $request, Aspiration $aspiration)
    {
        $request->validate([
            'status' => 'required|in:pending,process,done',
            'response' => 'nullable|string',
        ]);

        $aspiration->update($request->all());

        return redirect()->route('admin.aspirations.index')
            ->with('success', 'Aspirasi berhasil diperbarui');
    }

    public function destroy(Aspiration $aspiration)
    {
        $aspiration->delete();

        return redirect()->route('admin.aspirations.index')
            ->with('success', 'Aspirasi berhasil dihapus');
    }

    public function export(Request $request)
    {
        try {
            $status = $request->get('status');

            $query = Aspiration::orderBy('created_at', 'desc');

            if ($status) {
                $query->where('status', $status);
            }

            $aspirations = $query->get();

            $filename = 'aspirasi-dpm-' . date('Y-m-d-His') . '.csv';

            $headers = [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ];

            $callback = function () use ($aspirations) {
                $file = fopen('php://output', 'w');
                fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF));

                fputcsv($file, ['No', 'Nama', 'NIM', 'Email', 'Subjek', 'Pesan', 'Status', 'Respon', 'Tanggal']);

                $no = 1;
                foreach ($aspirations as $asp) {
                    fputcsv($file, [
                        $no++,
                        $asp->name,
                        $asp->nim ?? '-',
                        $asp->email,
                        $asp->subject,
                        $asp->message,
                        $this->getStatusLabel($asp->status),
                        $asp->response ?? '-',
                        $asp->created_at->format('d-m-Y H:i')
                    ]);
                }

                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
        } catch (\Exception $e) {
            return back()->with('error', 'Export gagal: ' . $e->getMessage());
        }
    }

    private function getStatusLabel($status)
    {
        $labels = [
            'pending' => 'Menunggu',
            'process' => 'Diproses',
            'done' => 'Selesai',
        ];

        return $labels[$status] ?? $status;
    }
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'nim' => 'required|string|max:20',
            'email' => 'required|email',
            'phone' => 'nullable|string|max:20',
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
        ]);
        $aspirasi = Aspirasi::create($request->all());

        // Kirim ke admin
        User::where('role', 'admin')
            ->each(function ($admin) use ($aspirasi) {
                $admin->notify(new AspirasiMasukNotification($aspirasi));
            });

        return back()->with('success', 'Aspirasi berhasil dikirim');
        $aspiration = Aspiration::create($validated);

        // Kirim email notifikasi ke pengirim
        Mail::to($aspiration->email)
            ->send(new AspirationSubmitted($aspiration));

        return redirect()->route('aspirations.create')
            ->with('success', 'Aspirasi Anda berhasil dikirim. Silakan cek email Anda.');
    }
}
